/*
 * adc_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/******************************************************************************
 *
 * The vADCTask task configures the ADC 0 peripheral of the TM4C1294NCPDT MCU
 * to sample on channel 0 (PE3).  It then configures uDMA to transfer the ADC
 * conversion data to buffers in a ping-pong fashion.  Therefore, there are
 * two buffers setup for the ping-pong mode.  The example sets up one task that
 * waits on a event group.  A event group is a group of notifications.  Any
 * matching notification values will unblock the BLOCKed task.  Once unblocked,
 * the task displays the average of the conversion values from each buffer on
 * the terminal window.  A hardware timer is also setup to trigger the ADC for
 * conversion repeatedly.
 *
 * The ISR for ADC 0 Sequence 0 is xADCSeq0Handler and it is linked to the
 * interrupt vector table for the TM4C1294NCPDT MCU in the startup_ccs.c file.
 * Instead of manually linking the ISR, the IntRegister API can be used
 * instead.  The ISR clears the hardware interrupt flag and then uses the
 * FreeRTOS Task Notify feature to defer further processing to a task.  This
 * makes for a very lean ISR which is recommended, especially for hardware
 * interrupts.
 *
 * The prvADCReceiveTask waits for the notification group from the ISR and then
 * begins to process the data from the buffer.  The data is summed
 * together to calculate the average and output the result over UART.
 *
 * This example uses AIN0 which is mapped to PE3 as the ADC input channel.
 * Supply the test voltage to this input.
 *
 * This example uses UARTprintf for output of UART messages.  UARTprintf is not
 * a thread-safe API and is only being used for simplicity of the demonstration
 * and in a controlled manner.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_adc.h"
#include "inc/hw_types.h"
#include "inc/hw_udma.h"
#include "driverlib/adc.h"
#include "driverlib/gpio.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/interrupt.h"
#include "driverlib/sysctl.h"
#include "driverlib/uart.h"
#include "driverlib/udma.h"
#include "utils/uartstdio.h"
/*-----------------------------------------------------------*/

/*
 * These are the event id for each half of the ADC ping-pong buffer
 * interrupt event.
 */
#define PING 0x1
#define PONG 0x2

/*
 * Define the size of the ADC buffers.
 */
#define ADC_SAMPLE_BUF_SIZE 256

/*
 * The control table used by the uDMA controller.  This table must be aligned
 * to a 1024 byte boundary.
 */
#if defined(ewarm)
#pragma data_alignment=1024
uint8_t pui8ControlTable[1024];
#elif defined(ccs)
#pragma DATA_ALIGN(pui8ControlTable, 1024)
uint8_t pui8ControlTable[1024];
#else
uint8_t pui8ControlTable[1024] __attribute__ ((aligned(1024)));
#endif

/*
 * The count of uDMA errors.  This value is incremented by the uDMA error
 * handler.
 */
static uint32_t g_ui32uDMAErrCount = 0;

/*
 * The count of times the uDMA interrupt occurred but the uDMA transfer was not
 * complete.  This should remain 0.
 */
static uint32_t g_ui32BadISR = 0;

/*
 * Global buffers to store ADC sample data.
 */
static uint16_t pui16ADCBuffer1[ADC_SAMPLE_BUF_SIZE];
static uint16_t pui16ADCBuffer2[ADC_SAMPLE_BUF_SIZE];

/*
 * Declare a variable that is used to hold the handle of the ADC
 * interrupt task.
 */
TaskHandle_t xADCSeq0Handle = NULL;

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvADCReceiveTask( void *pvParameters );

/*
 * Called by main() to create the ADC task.
 */
void vADCTask( void );

/* 
 * Hardware configuration for the ADC to configure channel, sequencer,
 * and interrupts.
 */
static void prvConfigureADC( void );
static void prvConfigureuDMA( void );
/*-----------------------------------------------------------*/

void vADCTask( void )
{
    /* Configure the ADC to use Channel 0 on Sequencer 0, be timer triggered,
     * and to fire an interrupt at the end of each sample sequence. */
    prvConfigureADC();

    /* Configure the uDMA for ADC 0 in ping-pong mode and point the uDMA to the
     * addresses of the defined ADC buffers to store ADC samples via the uDMA
     * ADC 0 channel. */
    prvConfigureuDMA();

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for ADC data processing task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvADCReceiveTask,
                 "ADC Receive",
                 configMINIMAL_STACK_SIZE + ADC_SAMPLE_BUF_SIZE*2,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 &xADCSeq0Handle );

    /* Initialize the UART and write initial status. */
    UARTprintf("Timer->ADC->uDMA demo!\n\n");
    UARTprintf("ui32AverageResult1\tui32AverageResult2\tTotal Samples\n");

    /* Enables DMA channel so it can perform transfers.  As soon as the
     * channels are enabled, the peripheral will issue a transfer request and
     * the data transfers will begin. */
    uDMAChannelEnable(UDMA_CHANNEL_ADC0);
}
/*-----------------------------------------------------------*/

static void prvADCReceiveTask( void *pvParameters )
{
unsigned long ulEventsToProcess;
uint32_t ui32Count, ui32AverageResult1, ui32AverageResult2;
uint32_t ui32SamplesTaken = 0;
uint32_t ulNotifiedValue;

    for( ;; )
    {
        /* Wait to receive a notification sent directly to this task from the
         * interrupt service routine.
         * The xTaskNotifyWait parameters in order are:
         *  - Which bits to clear on entry.
         *  - Which bits to clear on exit.
         *  - Storage location for the notified value.
         *  - How many RTOS ticks to wait for a notification. */
        ulEventsToProcess = xTaskNotifyWait( pdFALSE,
                                             PING | PONG,
                                             &ulNotifiedValue,
                                             portMAX_DELAY );

        if (ulEventsToProcess == pdPASS)
        {
            /* A notification was received.  See which bits were set. */
            if( ( ulNotifiedValue & PING ) == PING )

            {
                /* Process the data in pui16ADCBuffer1 and clear buffer
                 * entries. */
                ui32AverageResult1 = 0;

                for(ui32Count =0; ui32Count < ADC_SAMPLE_BUF_SIZE; ui32Count++)
                {
                    ui32AverageResult1 += pui16ADCBuffer1[ui32Count];
                }

                /* Enable for another uDMA block transfer. */
                uDMAChannelTransferSet(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT,
                                       UDMA_MODE_PINGPONG,
                                       (void *)(ADC0_BASE + ADC_O_SSFIFO0),
                                       &pui16ADCBuffer1, ADC_SAMPLE_BUF_SIZE);
                /* Enable DMA channel so it can perform transfers. */
                uDMAChannelEnable(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT);

                /* Track the number of samples taken and update the average. */
                ui32SamplesTaken += ADC_SAMPLE_BUF_SIZE;
                ui32AverageResult1 = ((ui32AverageResult1 ) /
                        ADC_SAMPLE_BUF_SIZE);

            }
            else if ( ( ulNotifiedValue & PONG ) == PONG )
            {
                /* Process the data in pui16ADCBuffer2 and clear buffer
                 * entries. */
                ui32AverageResult2 = 0;

                for(ui32Count =0; ui32Count < ADC_SAMPLE_BUF_SIZE; ui32Count++)
                {
                    ui32AverageResult2 += pui16ADCBuffer2[ui32Count];
                }

                /* Enable for another uDMA block transfer. */
                uDMAChannelTransferSet(UDMA_CHANNEL_ADC0 | UDMA_ALT_SELECT,
                                       UDMA_MODE_PINGPONG,
                                       (void *)(ADC0_BASE + ADC_O_SSFIFO0),
                                       &pui16ADCBuffer2, ADC_SAMPLE_BUF_SIZE);

                /* Enable DMA channel so it can perform transfers. */
                uDMAChannelEnable(UDMA_CHANNEL_ADC0 | UDMA_ALT_SELECT);

                /* Track the number of samples taken and update the average. */
                ui32SamplesTaken += ADC_SAMPLE_BUF_SIZE;
                ui32AverageResult2 = ((ui32AverageResult2 ) /
                        ADC_SAMPLE_BUF_SIZE);

                /* Both buffers have been filled by now, so print out the two
                 * average results and the number of samples taken so far. */
                UARTprintf("\t%4d\t\t\t%4d\t\t%d\r", ui32AverageResult1,
                           ui32AverageResult2, ui32SamplesTaken);
            }
        }
    }
}
/*-----------------------------------------------------------*/

static void prvConfigureADC( void )
{
    /* Enable the peripherals used by this application. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_ADC0);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);

    /* Enable the GPIO pin for ADC0 Channel 0 (PE3) and configure it for
     * analog functionality. */
    GPIOPinTypeADC(GPIO_PORTE_BASE, GPIO_PIN_3);

    /* Enable the peripherals used by this application. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_ADC0);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_ADC0))
    {
    }

    /* Disable interrupts for ADC0 sample sequence 0 to configure it. */
    ADCIntDisable(ADC0_BASE, 0);

    /* Disable ADC0 sample sequence 0.  With the sequence disabled, it is now
     * safe to load the new configuration parameters. */
    ADCSequenceDisable(ADC0_BASE, 0);

    /* Use ADC0 sequence 0 to sample channel 3 once for each timer period. */
    ADCClockConfigSet(ADC0_BASE, ADC_CLOCK_SRC_PIOSC | ADC_CLOCK_RATE_FULL, 1);

    /* Enable sample sequence 0 with a timer trigger.  Sequence 0 will start
     * the next conversion each time the timer times out. */
    ADCSequenceConfigure(ADC0_BASE, 0, ADC_TRIGGER_TIMER, 0);

    /* Configure step 0 on sequence 0.  Sample channel 0 (ADC_CTL_CH0) in
     * single-ended mode (default) and configure the interrupt flag
     * (ADC_CTL_IE) to be set when the sample is done.  Tell the ADC logic
     * that this is the last conversion on sequence 0 (ADC_CTL_END).  Sequence
     * 0 has 8 programmable steps.  Since we are only doing a single conversion
     * using sequence 0 we will only configure step 0.  For more information
     * on the ADC sequences and steps, reference the datasheet. */
    ADCSequenceStepConfigure(ADC0_BASE, 0, 0, ADC_CTL_CH0 | ADC_CTL_IE |
                             ADC_CTL_END);

    /* Since sample sequence 0 is now configured, it must be enabled. */
    ADCSequenceEnable(ADC0_BASE, 0);

    /* Clear the interrupt status flag.  This is done to make sure the
     * interrupt flag is cleared before we sample. */
    ADCIntClear(ADC0_BASE, 0);

    /* Enables the DMA channel for the ADC0 sample sequence 0. */
    ADCSequenceDMAEnable(ADC0_BASE, 0);

    /* Enable the ADC 0 sample sequence 0 interrupt. */
    ADCIntEnable(ADC0_BASE, 0);

    /* Enable the interrupt for ADC0 sequence 0 on the processor (NVIC). */
    IntEnable(INT_ADC0SS0);
}
/*-----------------------------------------------------------*/

static void prvConfigureuDMA( void )
{
    /* Enable the peripherals used by this application. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UDMA);

    /* Enable the uDMA controller. */
    uDMAEnable();

    /* Point at the control table to use for channel control structures. */
    uDMAControlBaseSet(pui8ControlTable);

    /* Put the attributes in a known state for the uDMA ADC0 channel.  These
     * should already be disabled by default. */
    uDMAChannelAttributeDisable(UDMA_CHANNEL_ADC0,
                                UDMA_ATTR_ALTSELECT | UDMA_ATTR_HIGH_PRIORITY |
                                UDMA_ATTR_REQMASK);

    /* Configure the control parameters for the primary control structure for
     * the ADC0 channel.  The primary control structure is used for the "A"
     * part of the ping-pong receive.  The transfer data size is 16 bits, the
     * source address does not increment since it will be reading from a
     * register.  The destination address increment is 16-bits.  The
     * arbitration size is set to one byte transfers. */
    uDMAChannelControlSet(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT, UDMA_SIZE_16 |
                          UDMA_SRC_INC_NONE | UDMA_DST_INC_16 | UDMA_ARB_1);

    /* Configure the control parameters for the alternate control structure for
     * the ADC0 channel.  The alternate control structure is used for the
     * "B" part of the ping-pong receive.  The configuration is identical to
     * the primary/A control structure. */
    uDMAChannelControlSet(UDMA_CHANNEL_ADC0 | UDMA_ALT_SELECT, UDMA_SIZE_16 |
                          UDMA_SRC_INC_NONE | UDMA_DST_INC_16 | UDMA_ARB_1);

    /* Set up the transfer parameters for the ADC0 primary control structure
     * The mode is set to ping-pong, the transfer source is the ADC Sample
     * Sequence Result FIFO 0 register, and the destination is the receive
     * "A" buffer.  The transfer size is set to match the size of the buffer.
     */
    uDMAChannelTransferSet(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT,
                           UDMA_MODE_PINGPONG,
                           (void *)(ADC0_BASE + ADC_O_SSFIFO0),
                           &pui16ADCBuffer1, ADC_SAMPLE_BUF_SIZE);

    /* Set up the transfer parameters for the ADC0 primary control structure
     * The mode is set to ping-pong, the transfer source is the ADC Sample
     * Sequence Result FIFO 0 register, and the destination is the receive
     * "B" buffer.  The transfer size is set to match the size of the buffer.
     */
    uDMAChannelTransferSet(UDMA_CHANNEL_ADC0 | UDMA_ALT_SELECT,
                           UDMA_MODE_PINGPONG,
                           (void *)(ADC0_BASE + ADC_O_SSFIFO0),
                           &pui16ADCBuffer2, ADC_SAMPLE_BUF_SIZE);

    /* Set the USEBURST attribute for the uDMA ADC0 channel.  This will force
     * the controller to always use a burst when transferring data from the
     * TX buffer to the UART.  This is somewhat more efficient bus usage than
     * the default which allows single or burst transfers. */
    uDMAChannelAttributeEnable(UDMA_CHANNEL_ADC0, UDMA_ATTR_USEBURST);
}
/*-----------------------------------------------------------*/

void xADCSeq0Handler( void )
{
    BaseType_t xADCTaskWoken;

    /* Clear the hardware interrupt flag for ADC0. */
    ADCIntClear(ADC0_BASE, 0);

    /* The xADCTaskWoken parameter must be initialized to pdFALSE as
     * it will get set to pdTRUE inside the interrupt safe API function if a
     * context switch is required. */
    xADCTaskWoken = pdFALSE;

    if (uDMAChannelModeGet(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT) ==
            UDMA_MODE_STOP)
    {

        /* Notify task to process the Ping buffer data. */
        xTaskNotifyFromISR(xADCSeq0Handle,
                           PING,
                           eSetBits,
                           &xADCTaskWoken);

        /* This FreeRTOS API call will handle the context switch if it is required
         * or have no effect if that is not needed. */
        portYIELD_FROM_ISR( xADCTaskWoken );
    }
    else if (uDMAChannelModeGet(UDMA_CHANNEL_ADC0 | UDMA_ALT_SELECT) ==
            UDMA_MODE_STOP)
    {

        /* Notify task to process the Pong buffer data. */
        xTaskNotifyFromISR(xADCSeq0Handle,
                           PONG,
                           eSetBits,
                           &xADCTaskWoken);

        /* This FreeRTOS API call will handle the context switch if it is required
         * or have no effect if that is not needed. */
        portYIELD_FROM_ISR( xADCTaskWoken );
    }
}
/*-----------------------------------------------------------*/

void
uDMAErrorHandler(void)
{
    /* The interrupt handler for uDMA errors.  This interrupt will occur if the
     * uDMA encounters a bus error while trying to perform a transfer.  This
     * handler just increments a counter if an error occurs. */

uint32_t ui32Status;

    /* Check for uDMA error bit */
    ui32Status = uDMAErrorStatusGet();

    /* If there is a uDMA error, then clear the error and increment
     * the error counter. */
    if(ui32Status)
    {
        uDMAErrorStatusClear();
        g_ui32uDMAErrCount++;
    }
}
/*-----------------------------------------------------------*/

void
uDMAIntHandler(void)
{
    /* The interrupt handler for uDMA interrupts from the memory channel.  This
     * interrupt will increment a counter, and then restart another memory
     * transfer. */

    g_ui32BadISR++;
}

